<?php
require_once __DIR__.'/../includes/config.php';
require_once __DIR__.'/../includes/auth.php';

// Set headers untuk response JSON dan CORS
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Error reporting hanya untuk development
if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
    ini_set('display_errors', 1);
    ini_set('display_startup_errors', 1);
    error_reporting(E_ALL);
}

try {
    // Handle OPTIONS request untuk CORS preflight
    if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
        http_response_code(200);
        exit();
    }

    // Pastikan method GET
    if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
        throw new Exception('Method not allowed', 405);
    }

    // Validasi session dan role
    if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'teacher') {
        throw new Exception('Unauthorized access - Teacher role required', 401);
    }

    // Validasi parameter exam_id
    if (!isset($_GET['exam_id']) || !is_numeric($_GET['exam_id'])) {
        throw new Exception('Invalid exam ID', 400);
    }

    $exam_id = (int)$_GET['exam_id'];
    $teacher_id = $_SESSION['user_id'];

    // Debug logging - pindahkan ke sini setelah $exam_id didefinisikan
    error_log("Generate token called with exam_id: " . $exam_id);
    error_log("User session: " . print_r($_SESSION, true));

    // Verifikasi ujian milik guru yang login
    $stmt = $pdo->prepare("SELECT id FROM exams WHERE id = ? AND teacher_id = ?");
    $stmt->execute([$exam_id, $teacher_id]);
    
    if (!$stmt->fetch()) {
        throw new Exception('Exam not found or access denied', 403);
    }

    // Generate token yang unik
    $token = bin2hex(random_bytes(32));
    $valid_until = date('Y-m-d H:i:s', strtotime('+10 minutes'));

    // Mulai transaction untuk keamanan data
    $pdo->beginTransaction();

    try {
        // Nonaktifkan token sebelumnya untuk exam yang sama
        $stmt = $pdo->prepare("UPDATE student_tokens SET is_active = 0 WHERE exam_id = ?");
        $stmt->execute([$exam_id]);

        // Simpan token baru ke database
        $stmt = $pdo->prepare("INSERT INTO student_tokens 
                             (token, exam_id, valid_until, is_active) 
                             VALUES (?, ?, ?, 1)");
        $success = $stmt->execute([$token, $exam_id, $valid_until]);

        if (!$success) {
            throw new Exception('Failed to save token to database', 500);
        }

        $pdo->commit();

        // Response sukses
        $response = [
            'success' => true,
            'token' => $token,
            'valid_until' => $valid_until,
            'exam_url' => BASE_URL . '/student/exam.php?token=' . $token,
            'message' => 'Token created successfully'
        ];

        echo json_encode($response);

    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }

} catch (Exception $e) {
    // Fix: Pastikan response code adalah integer
    $code = $e->getCode();
    if (!is_int($code) || $code < 400 || $code >= 600) {
        $code = 500;
    }
    http_response_code($code);
    
    $errorResponse = [
        'success' => false,
        'message' => $e->getMessage()
    ];

    // Tambahkan detail error hanya di environment development
    if (defined('ENVIRONMENT') && ENVIRONMENT === 'development') {
        $errorResponse['error_details'] = [
            'file' => $e->getFile(),
            'line' => $e->getLine(),
            'trace' => $e->getTraceAsString()
        ];
    }

    echo json_encode($errorResponse);
}